import os
import re
import json
import random
import string
from requests import Session
from bs4 import BeautifulSoup
from urllib.parse import urljoin
from .peewee import SqliteDatabase, Model, IntegerField, TextField, chunked
from warnings import simplefilter

simplefilter("ignore")
db = SqliteDatabase(None, pragmas={"foreign_keys": 1})


class BaseModel(Model):
    class Meta:
        database = db


class Game(BaseModel):
    game_pk = IntegerField(primary_key=True)
    official_date = TextField()
    game_date = TextField()
    away_team = TextField()
    home_team = TextField()
    content_link = TextField()


class LazyMLB:
    def __init__(self, cache_dir, config):
        self.mlb_api = "https://statsapi.mlb.com/api/v1/"
        self.lazy_api = "https://freesports.ddns.net/getM3U8.php"
        self.key_server = "freesports.ddns.net"
        self.s = Session()
        if "http_proxy" in config:
            self.s.proxies.update(
                {
                    "http": config["http_proxy"],
                    "https": config["http_proxy"],
                }
            )
        DB = os.path.join(cache_dir, "lazymlb0.db")
        db.init(DB)
        db.connect()
        db.create_tables(
            [Game],
            safe=True,
        )

    def fetch_teams(self):
        return

    def fetch_games_date(self, date):
        schedule_url = urljoin(self.mlb_api, "schedule")
        params = {"language": "en", "sportId": "1", "date": date}
        r = self.s.get(schedule_url, params=params)
        r.raise_for_status()
        res = r.json()

        def games(res):
            for date in res["dates"]:
                for game in date["games"]:
                    yield {
                        "game_pk": game["gamePk"],
                        "official_date": date["date"],
                        "game_date": game["gameDate"],
                        "away_team": game["teams"]["away"]["team"]["name"],
                        "home_team": game["teams"]["home"]["team"]["name"],
                        "content_link": game["content"]["link"],
                    }

        with db.atomic():
            for batch in chunked(games(res), 79):
                Game.replace_many(batch).execute()

    def get_games_date(self, date):
        if Game.select().where(Game.official_date == date).count() == 0:
            self.fetch_games_date(date)
        return Game.select().where(Game.official_date == date).order_by(Game.game_date)

    def get_game(self, game_pk):
        return Game.get(Game.game_pk == game_pk)

    def get_game_content(self, game_pk):
        game = Game.get(Game.game_pk == game_pk)
        content_url = urljoin(self.mlb_api, game.content_link)
        r = self.s.get(content_url)
        r.raise_for_status()
        return r.json()

    def get_media(self, date, media_id):
        params = {
            "league": "MLB",
            "date": date,
            "id": media_id,
            "cdn": "akc",
        }
        r = self.s.get(self.lazy_api, params=params, verify=False)
        r.raise_for_status()
        return r.text

    def get_media_roy(self, date, team, feed):
        start = "http://roystream.com/c/baseball/mlb"
        headers = {
            "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/103.0.0.0 Safari/537.36",
            "Accept": "text/html, text/plain, */*",
            "Accept-Language": "en-US,en;q=0.9",
            "Referer": "http://roystream.com/",
        }
        r = self.s.get(start, headers=headers)
        r.raise_for_status()
        soup = BeautifulSoup(r.content, "html.parser", from_encoding="utf-8")
        for game in soup.select(f"[data-date-started='{date}']"):
            if game.parent.find(text=re.compile(team, re.I)):
                game_url = urljoin(start, game.parent.a.get("href"))
                break
        else:
            return
        r = self.s.get(game_url, headers=headers)
        r.raise_for_status()
        soup = BeautifulSoup(r.content, "html.parser", from_encoding="utf-8")
        piframes = json.loads(soup.select_one("#pIframes").get("value"))
        for stream in piframes:
            if stream["title"].lower() == feed.lower():
                stream_url = (
                    "http://roystream.com/stream/"
                    + stream["slug"]
                    + ".html?t="
                    + "".join([random.choice(string.ascii_lowercase + string.digits) for i in range(7)])
                )
                break
        else:
            return
        r = self.s.get(stream_url, headers=headers)
        r.raise_for_status()
        soup = BeautifulSoup(r.content, "html.parser", from_encoding="utf-8")
        search_m3u8 = re.search(r"url:\s*'.*(https[^']+)'", r.text)
        if search_m3u8:
            _m3u8 = search_m3u8.group(1)
        else:
            return
        search_key = re.search(r"replace\([\"']+([^\"']{10,})[\"']+,\s*[\"']+([^\"']{10,})", r.text)
        if search_key:
            _key = [search_key.groups()]
        else:
            return
        return (_m3u8, _key)

    def fix66(self, url):
        re_fix = re.match("(.*)(hmac=|hash=)([1-9a-egh]+)(/.*)", url)
        if re_fix:
            n = re_fix.groups()
            return n[0] + n[1] + n[2].replace("g", "f").replace("h", "0") + n[3]
        else:
            return url

    def get_media66(self, media_id):
        headers = {
            "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/103.0.0.0 Safari/537.36",
            "Accept": "application/json, text/plain, */*",
            "Accept-Language": "en-US,en;q=0.9",
        }
        r = self.s.get("https://api.mlb66.ir/api/sport/schedule", headers=headers)
        r.raise_for_status()
        for game in r.json()["games"]:
            for stream in game["streams"]:
                if stream["mediaid"] == str(media_id):
                    if stream["url"]:
                        r = self.s.get(self.fix66(stream["url"]), headers=headers, stream=True)
                        r.raise_for_status()
                        return (self.fix66(stream["url"]), stream["keys"])
                    else:
                        r = self.s.get(
                            "https://api.mlb66.ir/api/sport/publish_point/{0}".format(stream["sharestream_uid"]),
                            headers=headers,
                        )
                        return (r.json(), stream["keys"])
